program LUFACTOR;
{--------------------------------------------------------------------}
{  Alg3'3.pas   Pascal program for implementing Algorithm 3.3        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 3.3 (PA = LU Factorization with Pivoting).              }
{  Section   3.6, Triangular Factorizaton, Page 175                  }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxR = 10;
    Epsilon = 4E-12;
  type
    SubS = 1..MaxR;
    VECTOR = array[SubS] of real;
    MATRIX = array[SubS, SubS] of real;
    POINTER = array[SubS] of integer;
    LETTER = string[4];
    LETTERS = string[200];
    Status = (Done, Working);
    DoSome = (Go, New, Stop);
    PivStr = (Trivial, Partial);
    MatType = (LowerT, Square, UpperT);

  var
    Hor, InRC, Inum, N, Sub, Ver: integer;
    B, DX, X: VECTOR;
    A, A1, C: MATRIX;
    Row: POINTER;
    Cond, Det, MaxA, MaxC, Rnum: real;
    Ans: CHAR;
    Ach, Bch: LETTER;
    Mess: LETTERS;
    Stat: Status;
    DoMo: DoSome;
    Pmeth: PivStr;
    Mtype: MatType;

  procedure FACTOR (var A: MATRIX; var Row: POINTER; N: integer; var Det: real);
    label
      999;
    var
      C, J, K, P, RowK, RowP, T: integer;
  begin
    Det := 1;
    for J := 1 to N do                      {Initialize Pointer Vector}
      Row[J] := J;
    for P := 1 to N - 1 do                 {Upper Triangularization Loop}
      begin
        if Pmeth = Trivial then                      {Find the pivot row}
          begin                                  {using trivial pivoting}
            for K := P + 1 to N do
              begin
                if A[Row[P], P] = 0 then
                  begin
                    T := Row[P];
                    Row[P] := Row[K];
                    Row[K] := T;
                    Det := -Det;
                  end;
              end;
          end;
        if Pmeth = Partial then                      {Find the pivot row}
          begin                                  {using partial pivoting}
            for K := P + 1 to N do
              begin
                if ABS(A[Row[K], P]) > ABS(A[Row[P], P]) then
                  begin
                    T := Row[P];
                    Row[P] := Row[K];
                    Row[K] := T;
                    Det := -Det;
                  end;
              end;
          end;
        Det := Det * A[Row[P], P];
        if Det = 0 then                         {Check Singular Matrix}
          goto 999;
        for K := P + 1 to N do                     {Gaussian Elimination}
          begin
            RowK := Row[K];
            RowP := Row[P];
            A[RowK, P] := A[RowK, P] / A[RowP, P];
            for C := P + 1 to N do
              A[RowK, C] := A[RowK, C] - A[RowK, P] * A[RowP, C];
          end;                                 {End Gaussian Elimination}
      end;                                  {End Upper Triangularization}
999:
    Det := Det * A[Row[N], N];
  end;                                        {End of procedure FACTOR}

  procedure SOLVE (A: MATRIX; Row: POINTER; B: VECTOR; var X: VECTOR; N: integer);
    label
      999;
    var
      C, K, RowK: integer;
      Sum: real;
  begin
    for K := 1 to N do
      if A[Row[K], K] = 0 then                   {Check Singular Matrix}
        goto 999;
    X[1] := B[Row[1]];                             {Forward Substitution}
    for K := 2 to N do
      begin
        Sum := 0;
        RowK := Row[K];
        for C := 1 to K - 1 do
          Sum := Sum + A[RowK, C] * X[C];
        X[K] := B[RowK] - Sum;                   {End Forward Substitution}
      end;
    X[N] := X[N] / A[Row[N], N];                         {Back Substitution}
    for K := N - 1 downto 1 do
      begin
        Sum := 0;
        RowK := Row[K];
        for C := K + 1 to N do
          Sum := Sum + A[RowK, C] * X[C];
        X[K] := (X[K] - Sum) / A[RowK, K];
      end;
999:
  end;                                         {End of procedure SOLVE}

  procedure INVERSE (A: MATRIX; Row: POINTER; var C: MATRIX; N: integer);
    var
      J, K: integer;
      B, X: VECTOR;
  begin
    for K := 1 to N do
      begin
        for J := 1 to N do
          begin
            if J = K then
              B[J] := 1
            else
              B[J] := 0;
          end;
        SOLVE(A, Row, B, X, N);
        for J := 1 to N do
          C[J, K] := X[J];
      end;
  end;

  procedure ERRORS (A, A1: MATRIX; var C: MATRIX; Row: POINTER; B, X: VECTOR; var DX: VECTOR; N: integer);
    var
      J, K: integer;
      Ajk, Max, Sum, SumA, SumC, Term: real;
      E: VECTOR;
  begin
    INVERSE(A, Row, C, N);
    for J := 1 to N do
      begin
        Sum := 0;
        Max := 0;
        for K := 1 to N do
          begin
            Term := A1[J, K] * X[K];
            Sum := Sum + Term;
            if Max < ABS(Term) * Epsilon then
              Max := ABS(Term) * Epsilon;
            if Max = 0 then
              Max := ABS(A1[J, K]) * Epsilon;
          end;
        if Max < ABS(B[J]) * Epsilon then
          Max := ABS(B[J]) * Epsilon;
        E[J] := ABS(Sum - B[J]);
        if E[J] < Max then
          E[J] := Max / (N + 1);
      end;
    for J := 1 to N do
      begin
        Sum := 0;
        for K := 1 to N do
          begin
            Sum := Sum + ABS(C[J, K]) * E[K];
          end;
        DX[J] := Sum;
      end;
    MaxA := 0;
    MaxC := 0;
    for K := 1 to N do
      begin
        SumA := 0;
        SumC := 0;
        for J := 1 to N do
          begin
            SumA := SumA + ABS(A1[J, K]);
            SumC := SumC + ABS(C[J, K]);
          end;
        if MaxA < SumA then
          MaxA := SumA;
        if MaxC < SumC then
          MaxC := SumC;
      end;
    Cond := MaxA * MaxC;
  end;

  procedure INPUTMATRIX (var Ach: LETTER; var A, A1: MATRIX; N, InRC: integer);
    var
      Count, C, CL, CU, K, R, RL, RU: integer;
      Z: VECTOR;
  begin
    for R := 1 to N do
      begin
        for C := 1 to N do
          begin
            A[R, C] := 0;
            A1[R, C] := A[R, C];
          end;
      end;
    WRITELN;
    WRITELN('     Input the elements of the ', N : 1, ' by ', N : 1, ' coefficient matrix  ', Ach);
    RL := 1;
    RU := N;
    CL := 1;
    CU := N;
    if (InRC = 1) then
      begin
        for R := 1 to N do
          begin
            WRITELN;
            WRITELN('ENTER all the coefficients of row ', R, ' on one row');
            WRITELN;
            for K := 1 to N do
              Z[K] := 0;
            case N of
              1: 
                READLN(Z[1]);
              2: 
                READLN(Z[1], Z[2]);
              3: 
                READLN(Z[1], Z[2], Z[3]);
              4: 
                READLN(Z[1], Z[2], Z[3], Z[4]);
              5: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5]);
              6: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6]);
              7: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7]);
              8: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8]);
              9: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9]);
              10: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6], Z[7], Z[8], Z[9], Z[10]);
            end;
            for C := 1 to N do
              begin
                A[R, C] := Z[C];
                A1[R, C] := A[R, C];
              end;
          end;
      end
    else if (InRC = 2) then
      begin
        for R := 1 to N do
          begin
            WRITELN;
            WRITELN('     ENTER the coefficients of row ', R);
            WRITELN;
            if Mtype = LowerT then
              CU := R;
            if Mtype = UpperT then
              CL := R;
            for C := CL to CU do
              begin
                WRITELN;
                WRITE('     A(', R, ',', C, ') = ');
                READLN(A[R, C]);
                A1[R, C] := A[R, C];
              end;
          end;
      end
    else
      begin
        for C := 1 to N do
          begin
            WRITELN;
            WRITELN('     ENTER the coefficients of column ', C);
            WRITELN;
            if Mtype = LowerT then
              RL := C;
            if Mtype = UpperT then
              RU := C;
            for R := RL to RU do
              begin
                WRITELN;
                WRITE('     A(', R, ',', C, ') = ');
                READLN(A[R, C]);
                A1[R, C] := A[R, C];
              end;
          end;
      end;
    Mtype := Square;
  end;                                   {End of procedure INPUTMATRIX}

  procedure REFRESH (var A: MATRIX; A1: MATRIX; N: integer);
    var
      C, R: integer;
  begin
    for R := 1 to N do
      begin
        for C := 1 to N do
          begin
            A[R, C] := A1[R, C];
          end;
      end;
  end;

  procedure Aoutput (Ach: LETTER; A: MATRIX; N: integer);
    var
      Digits, Mdigits, C, R: integer;
      Log10: real;
  begin
    Log10 := LN(10);
    WRITELN;
    WRITELN('The matrix  ', Ach, '  is:');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to N - 1 do
          begin
            Digits := 7;
            if A[R, C] <> 0 then
              Mdigits := 1 + TRUNC(LN(ABS(A[R, C])) / Log10);
            if A[R, C] < 0 then
              Mdigits := Mdigits + 1;
            if Mdigits < 7 then
              Mdigits := 7;
            Digits := 14 - Mdigits;
            WRITE(A[R, C] : 15 : Digits, ' ');
          end;
        Digits := 7;
        if A[R, N] <> 0 then
          Mdigits := 1 + TRUNC(LN(ABS(A[R, N])) / Log10);
        if A[R, N] < 0 then
          Mdigits := Mdigits + 1;
        if Mdigits < 7 then
          Mdigits := 7;
        Digits := 14 - Mdigits;
        WRITE(A[R, N] : 15 : Digits);
        if N > 5 then
          WRITELN;
      end;
    WRITELN;
  end;                                       {End of procedure Aoutput}

  procedure Boutput (Bch: LETTER; B: VECTOR; N: integer);
    var
      J: integer;
  begin
    WRITELN;
    WRITELN('     The Vector  ', Bch, '  is:');
    WRITELN;
    for J := 1 to N do
      begin
        WRITELN('     B(', J : 2, ') =', B[J] : 15 : 7);
      end;
    WRITELN;
  end;                                      {End of procedure BXoutput}

  procedure PrintFactors (A, A1: MATRIX; Row: POINTER; N: integer);
    var
      C, R: integer;
      A0: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('The Matrix  P*A  is:');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to N do
          WRITE(A1[Row[R], C] : 15 : 8, ' ');
      end;
    WRITELN;
    WRITELN;
    WRITELN('The Matrix  L  is:');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to R - 1 do
          WRITE(A[Row[R], C] : 15 : 8, ' ');
        A0 := 1;
        WRITE(A0 : 15 : 8, ' ');
        for C := R + 1 to N do
          begin
            A0 := 0;
            WRITE(A0 : 15 : 8, ' ');
          end;
      end;
    WRITELN;
    WRITELN;
    WRITELN('The Matrix  U  is:');
    for R := 1 to N do
      begin
        WRITELN;
        for C := 1 to R - 1 do
          begin
            A0 := 0;
            WRITE(A0 : 15 : 8, ' ');
          end;
        for C := R to N do
          WRITE(A[Row[R], C] : 15 : 8, ' ');
      end;
    WRITELN;
  end;                                  {End of procedure PrintFactors}

  procedure CHANGEMATRIX (Ach: LETTER; var A, A1: MATRIX; N: integer);
    type
      STATUS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      Count, C, I, K, R: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) or (Stat = Bad) do
      begin
        CLRSCR;
        Aoutput(Ach, A1, N);
        WRITELN;
        if (Stat <> Bad) then
          begin
            WRITE('Do you want to make a change in the matrix ? <Y/N> ');
            READLN(Resp);
          end;
        if (Resp = 'Y') or (Resp = 'y') or (Stat = Bad) then
          begin
            WRITELN;
            WRITELN('     To change a coefficient select');
            case N of
              2: 
                begin
                  WRITELN('        the row    R = 1,2');
                  WRITELN('        and column C = 1,2');
                end;
              3: 
                begin
                  WRITELN('        the row    R = 1,2,3');
                  WRITELN('        and column C = 1,2,3');
                end;
              else
                begin
                  WRITELN('        the row    R = 1,2,...,', N : 2);
                  WRITELN('        and column C = 1,2,...,', N : 2);
                end;
            end;
            Mess := '     ENTER the row R = ';
            R := 0;
            WRITE(Mess);
            READLN(R);
            Mess := '     ENTER column  C = ';
            C := 0;
            WRITE(Mess);
            READLN(C);
            if (1 <= R) and (R <= N) and (1 <= C) and (C <= N) then
              begin
                WRITELN;
                WRITELN('     The current value is A(', R : 2, ',', C : 2, ') =', A[R, C] : 15 : 7);
                WRITELN;
                WRITE('     ENTER the NEW value  A(', R : 2, ',', C : 2, ') = ');
                READLN(A[R, C]);
                A1[R, C] := A[R, C];
                WRITELN;
              end;
          end
        else
          Stat := Done;
      end;
  end;

  procedure CHANGEVECTOR (Bch: LETTER; var B: VECTOR; N: integer);
    type
      STATUS = (Enter, Done);
      LETTER = string[1];
    var
      Count, C, I, K, R: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) do
      begin
        CLRSCR;
        WRITELN;
        Boutput(Bch, B, N);
        WRITELN;
        WRITE('     Do you want to make a change in the vector ? <Y/N> ');
        READLN(Resp);
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            WRITELN;
            WRITE('     To change a coefficient select the row ');
            case N of
              2: 
                WRITELN('R = 1,2');
              3: 
                WRITELN('R = 1,2,3');
              else
                WRITELN('R = 1,2,...,', N : 2);
            end;
            WRITELN;
            Mess := '                              ENTER the row R = ';
            R := 0;
            WRITE(Mess);
            READLN(R);
            if (1 <= R) and (R <= N) then
              begin
                WRITELN;
                WRITELN('     The current value is B(', R : 2, ') =', B[R] : 15 : 7);
                WRITELN;
                WRITE('     ENTER the NEW value  B(', R : 2, ') = ');
                READLN(B[R]);
                WRITELN;
              end;
          end
        else
          Stat := Done;
      end;
  end;

  procedure STRATEGY (var Pmeth: PivStr);
    var
      I: integer;
      Ans: LETTER;
  begin
    CLRSCR;
    for I := 1 to 7 do
      WRITELN;
    WRITELN('          You have the choice of two pivoting strategies.');
    WRITELN;
    WRITELN;
    WRITELN('          < 1 > Trivial pivoting');
    WRITELN;
    WRITELN;
    WRITELN('          < 2 > Partial pivoting');
    WRITELN;
    WRITELN;
    Mess := '          SELECT your strategy  < 1 or 2 > ?  ';
    I := 2;
    WRITE(Mess);
    READLN(I);
    if I = 1 then
      Pmeth := Trivial
    else
      Pmeth := Partial;
  end;

  procedure MESSAGE (var InRC: integer; var Pmeth: PivStr; var Mtype: MatType);
    var
      I: integer;
      Ans: LETTER;
  begin
    CLRSCR;
    WRITELN('                      TRIANGULAR FACTORIZATION');
    WRITELN;
    WRITELN;
    WRITELN('          Solution of the linear system  A*X = B.');
    WRITELN;
    WRITELN('     The method is  LU  factorization with row interchanges.');
    WRITELN;
    WRITELN('     If the matrix  A  is nonsingular, then a permutation matrix  P,');
    WRITELN;
    WRITELN('     a lower-triangular matrix  L, and an upper-triangular matrix U');
    WRITELN;
    WRITELN('     are constructed so that P*A = L*U.  Then the system  L*Y = P*B');
    WRITELN;
    WRITELN('     is solved for  Y,  and the system  U*X = Y  is solved for  X.');
    WRITELN;
    WRITELN('     The matrices P, L and U are retained.  If the system  A*X = B');
    WRITELN;
    WRITELN('     is to be solved again, but with a different vector  B,  then');
    WRITELN;
    WRITELN('     it will take less computing effort the second time.');
    WRITELN;
    WRITELN;
    WRITE('                      Press the <ENTER> key.  ');
    READLN(Ans);
    CLRSCR;
    WRITELN;
    WRITELN('          What type of matrix will you enter ?');
    WRITELN;
    WRITELN;
    WRITELN('          < 1 > An N by N square matrix.');
    WRITELN;
    WRITELN;
    WRITELN('          < 2 > An N by N lower-triangular matrix.');
    WRITELN;
    WRITELN;
    WRITELN('          < 3 > An N by N upper-triangular matrix.');
    WRITELN;
    WRITELN;
    Mess := '          SELECT the type of matrix  < 1 - 3 > ?  ';
    I := 1;
    WRITE(Mess);
    READLN(I);
    if (I < 1) or (3 < I) then
      I := 1;
    if I = 1 then
      Mtype := Square;
    if I = 2 then
      Mtype := LowerT;
    if I = 3 then
      Mtype := UpperT;
    STRATEGY(Pmeth);
    CLRSCR;
    WRITELN;
    WRITELN('        Choose how you want to input the elements of the matrix.');
    WRITELN;
    WRITELN('    <1> Enter the elements of each row on one line separated by spaces, i.e.');
    WRITELN;
    WRITELN('        A(J,1)  A(J,2)  ...  A(J,N)           for J=1,2,...,N');
    WRITELN;
    WRITELN('    <2> Enter each element of a row on a separate line, i.e.');
    WRITELN;
    WRITELN('        A(J,1)');
    WRITELN('        A(J,2)');
    WRITELN('           .');
    WRITELN('           :');
    WRITELN('        A(J,N)     for J=1,2,...,N');
    WRITELN;
    WRITELN('    <3> Enter each element of a column on a separate line, i.e.');
    WRITELN;
    WRITELN('        A(1,K)');
    WRITELN('        A(2,K)');
    WRITELN('           .');
    WRITELN('           :');
    WRITELN('        A(N,K)     for K=1,2,...,N');
    WRITELN;
    Mess := '        SELECT <1 - 3> ?  ';
    InRC := 3;
    WRITE(Mess);
    READLN(InRC);
    if (InRC <> 1) and (InRC <> 2) and (InRC <> 3) then
      InRC := 2;
  end;                                  {End of procedure MESSAGE}

  procedure INPUTS (var A, A1: MATRIX; var N, InRC: integer);
    var
      C, I, R: integer;
  begin
    CLRSCR;
    for I := 1 to 6 do
      WRITELN;
    WRITELN('        Solution of the linear system of equations  A*X = B.');
    WRITELN;
    WRITELN('               A  is a matrix of dimension  N by N.');
    WRITELN;
    WRITELN('               B  is an  N  dimensional vector.');
    WRITELN;
    WRITELN('              {N  must be an integer between 1 and 10}');
    WRITELN;
    Mess := '        ENTER  N  =  ';
    N := 2;
    WRITE(Mess);
    READLN(N);
    if (N < 2) then
      N := 2;
    if (N > 10) then
      N := 10;
    CLRSCR;
    Ach := 'A';
    INPUTMATRIX(Ach, A, A1, N, InRC);
  end;                                   {End of procedure INPUTS}

  procedure INPUTVECTOR (Bch: LETTER; var B: VECTOR; N: integer);
    var
      R: integer;
  begin
    for R := 1 to N do
      B[R] := 0;
    CLRSCR;
    WRITELN;
    WRITELN;
    Aoutput(Ach, A1, N);
    WRITELN;
    WRITELN('Enter the Column Vector ', Bch, ' .');
    WRITELN;
    for R := 1 to N do                               {Input Vector B}
      begin
        WRITELN;
        WRITE('     B(', R : 2, ') = ');
        READLN(B[R]);
      end;
  end;                                   {End of procedure INPUTVECTOR}

  procedure BXoutput (B, X, DX: VECTOR; Cond, Det: real; N: integer);
    var
      J: integer;
  begin
    WRITELN;
    WRITELN('Column Vector B(J):      Solution Vector X(J):         Error Vector:');
    for J := 1 to N do
      begin
        WRITELN;
        WRITE('B(', J : 2, ') =', B[J] : 15 : 8, '    ');
        WRITE('X(', J : 2, ') =', X[J] : 15 : 8);
        WRITE('  +-', DX[J] : 15 : 8);
      end;
    WRITELN;
    WRITELN;
    WRITELN('     The determinant of the matrix A is ', Det : 15 : 8);
    WRITELN;
    WRITELN('The condition number of the matrix A is ', Cond);
  end;                                      {End of procedure BXoutput}

  procedure DOMORE (var Stat: Status);
    var
      Resp: CHAR;
  begin
    WRITELN;
    WRITELN;
    WRITE('Want to solve A*X = B with a new vector  B ? <Y/N> ');
    READLN(Resp);
    if (Resp <> 'y') and (Resp <> 'Y') then
      Stat := Done;
  end;                                        {End of procedure DOMORE}

begin                                            {Begin Main Program}
  MESSAGE(InRC, Pmeth, Mtype);
  DoMo := Go;
  while (DoMo = Go) or (DoMo = New) do
    begin
      if DoMo = Go then
        begin
          N := MaxR;
          INPUTS(A, A1, N, InRC)
        end
      else
        begin
          WRITELN;
          WRITELN;
          WRITE('     Want a completely new matrix ? <Y/N> ');
          READLN(Ans);
          if (Ans = 'Y') or (Ans = 'y') then
            INPUTS(A, A1, N, InRC)
          else
            REFRESH(A, A1, N);
          WRITELN;
          WRITELN;
          WRITE('Use a different pivoting strategy ? <Y/N>  ');
          READLN(Ans);
          if (Ans = 'Y') or (Ans = 'y') then
            STRATEGY(Pmeth);
        end;
      CHANGEMATRIX(Ach, A, A1, N);
      FACTOR(A, Row, N, Det);
      Stat := Working;
      while Stat = Working do
        begin
          CLRSCR;
          Aoutput(Ach, A1, N);
          INPUTVECTOR('B', B, N);
          CHANGEVECTOR('B', B, N);
          if Det <> 0 then
            begin
              SOLVE(A, Row, B, X, N);
              ERRORS(A, A1, C, Row, B, X, DX, N);
            end;
          CLRSCR;
          Aoutput(Ach, A1, N);
          if Det = 0 then
            begin
              WRITELN;
              WRITELN('The Matrix  ', Ach, '  is singular.');
              WRITELN;
              Stat := Done;
            end
          else
            BXoutput(B, X, DX, Cond, Det, N);
          if Stat <> Done then
            DOMORE(Stat);
        end;
      WRITELN;
      WRITELN;
      WRITE('Want to find see the factors P*A, L and  U ? <Y/N> ');
      READLN(Ans);
      if (Ans = 'Y') or (Ans = 'y') then
        PrintFactors(A, A1, Row, N);
      WRITELN;
      WRITELN;
      WRITE('Want  to find  the inverse  A^-1  ? <Y/N>  ');
      READLN(Ans);
      if (Ans = 'Y') or (Ans = 'y') then
        begin
          if Det <> 0 then
            begin
              CLRSCR;
              Aoutput(Ach, A1, N);
              Aoutput('A^-1', C, N);
            end
          else
            begin
              WRITELN;
              WRITELN('A singular matrix does not have an inverse.');
            end;
        end;
      WRITELN;
      WRITELN;
      WRITE('Do you want to solve a new system ? <Y/N>  ');
      READLN(Ans);
      if (Ans = 'Y') or (Ans = 'y') then
        DoMo := New
      else
        DoMo := Stop;
    end;
end.                                            {End of Main Program}

